"""
Classic Bomberman Game
How to run: python classic_bomberman.py

Controls:
Player 1 (Red): WASD to move, Space to place bomb
Player 2 (Blue): Arrow keys to move, Enter to place bomb  
Player 3 (Green): TFGH to move, R to place bomb
Player 4 (Yellow): IJKL to move, U to place bomb

Game Rules:
- Bombs explode in cross-shaped flames
- Flames can destroy soft walls with 20% chance to drop items
- Items include: Fire power up, Bomb count up, Speed up
- Players caught in flames can be rescued by teammates
- Last player standing wins

Dependencies: pygame 2.x, standard library
Resolution: 832 × 704 pixels (13 cols × 11 rows, 64px per tile)
Frame Rate: 60 FPS
"""

import pygame
import random
import time
import math
from dataclasses import dataclass, field
from enum import Enum
from typing import List, Dict, Tuple, Optional, Set
import os
import sys
from game_difficulty import DifficultyLevel, get_difficulty_config

# 初始化Pygame
pygame.init()

# 常量定义
WINDOW_WIDTH = 832  # 13 * 64
WINDOW_HEIGHT = 704  # 11 * 64
GRID_WIDTH = 13
GRID_HEIGHT = 11
TILE_SIZE = 64
FPS = 60
MOVE_DELAY = 8  # 添加移动延迟帧数，值越大移动越慢

# 颜色定义
BLACK = (0, 0, 0)
WHITE = (255, 255, 255)
RED = (255, 0, 0)
BLUE = (0, 0, 255)
GREEN = (0, 255, 0)
YELLOW = (255, 255, 0)
GRAY = (128, 128, 128)
DARK_GRAY = (64, 64, 64)
ORANGE = (255, 165, 0)
PURPLE = (128, 0, 128)
BROWN = (139, 69, 19)
LIGHT_BLUE = (173, 216, 230)

class Tile(Enum):
    """地图瓦片类型枚举"""
    FLOOR = 0      # 地板
    SOFT = 1       # 软墙(可破坏)
    HARD = 2       # 硬墙(不可破坏)
    BOMB = 3       # 炸弹
    ITEM = 4       # 道具

class Item(Enum):
    """道具类型枚举"""
    FIRE_UP = 0    # 火力增强
    BOMB_UP = 1    # 炸弹数量增加
    SPEED_UP = 2   # 速度提升

@dataclass
class Vector2:
    """二维向量类"""
    x: int
    y: int
    
    def __add__(self, other):
        return Vector2(self.x + other.x, self.y + other.y)
    
    def __eq__(self, other):
        return self.x == other.x and self.y == other.y

@dataclass
class Player:
    """玩家数据类"""
    pos: Vector2                    # 网格坐标
    fire: int = 1                   # 爆炸范围
    bombs: int = 1                  # 同时炸弹数量限制
    speed: int = 1                  # 移动速度
    alive: bool = True              # 是否存活
    trapped_ticks: int = 0          # 被困住的剩余帧数
    input_scheme: Dict[str, int] = field(default_factory=dict)  # 输入方案
    color: Tuple[int, int, int] = WHITE  # 玩家颜色
    active_bombs: int = 0           # 当前活跃炸弹数量
    player_id: int = 0              # 玩家ID
    move_cooldown: int = 0          # 移动冷却时间

@dataclass
class Bomb:
    """炸弹数据类"""
    owner_id: int                   # 拥有者ID
    grid_pos: Vector2               # 网格位置
    timer: int = 300                # 倒计时(5秒 * 60FPS)，在Gym环境中会被替换为步数
    fire: int = 1                   # 爆炸范围

@dataclass
class ItemDrop:
    """掉落道具数据类"""
    item_type: Item                 # 道具类型
    grid_pos: Vector2               # 网格位置

class AssetManager:
    """Asset Manager - handles image and sound loading"""
    
    def __init__(self):
        self.images = {}
        self.sounds = {}
        # Extended asset path list
        self.asset_paths = [
            # Bomb related
            "game/assets-necessay/kenney/3D assets/Platformer Kit/Previews/bomb.png",
            "game/assets-necessay/kenney/2D assets/Platformer Assets Base/PNG/Items/bomb.png",
            "game/assets-necessay/kenney/2D assets/Platformer Assets Base/PNG/Items/bombFlash.png",
            "game/assets-necessay/kenney/2D assets/Platformer Pack Redux/PNG/Tiles/bomb.png",
            "game/assets-necessay/kenney/2D assets/Platformer Pack Redux/PNG/Tiles/bombWhite.png",
            # Character related
            "game/assets-necessay/kenney/3D assets/Mini Arcade/Previews/character-gamer.png",
            "game/assets-necessay/kenney/3D assets/Mini Arcade/Previews/character-employee.png",
            "game/assets-necessay/kenney/3D assets/Blocky Characters/Faces/face_robot.png",
            "game/assets-necessay/kenney/2D assets/Abstract Platformer/PNG/Players/Player Green/playerGreen_stand.png",
            "game/assets-necessay/kenney/2D assets/Abstract Platformer/PNG/Players/Player Blue/playerBlue_stand.png",
            "game/assets-necessay/kenney/2D assets/Abstract Platformer/PNG/Players/Player Grey/playerGrey_stand.png",
            # Explosion effects
            "game/assets-necessay/kenney/2D assets/Explosion Pack/PNG/Regular explosion/regularExplosion00.png",
            "game/assets-necessay/kenney/2D assets/Explosion Pack/PNG/Regular explosion/regularExplosion01.png",
            "game/assets-necessay/kenney/2D assets/Explosion Pack/PNG/Regular explosion/regularExplosion02.png",
            "game/assets-necessay/kenney/2D assets/Explosion Pack/PNG/Regular explosion/regularExplosion03.png",
            "game/assets-necessay/kenney/2D assets/Smoke Particles/PNG/Explosion/explosion00.png",
            "game/assets-necessay/kenney/Icons/Board Game Icons/PNG/Double (128px)/fire.png",
            "game/assets-necessay/kenney/2D assets/Particle Pack/PNG (Transparent)/fire_01.png",
            # Blocks and walls
            "game/assets-necessay/kenney/2D assets/Sokoban Pack/PNG/Default size/Blocks/block_01.png",
            "game/assets-necessay/kenney/2D assets/Sokoban Pack/PNG/Default size/Blocks/block_02.png",
            "game/assets-necessay/kenney/2D assets/Sokoban Pack/PNG/Default size/Blocks/block_03.png",
            "game/assets-necessay/kenney/2D assets/Sokoban Pack/PNG/Default size/Blocks/block_04.png",
            "game/assets-necessay/kenney/2D assets/Sokoban Pack/PNG/Default size/Blocks/block_05.png",
            "game/assets-necessay/kenney/2D assets/Sokoban Pack/PNG/Default size/Blocks/block_06.png",
            "game/assets-necessay/kenney/2D assets/Sokoban Pack/PNG/Default size/Blocks/block_07.png",
            "game/assets-necessay/kenney/2D assets/Sokoban Pack/PNG/Default size/Blocks/block_08.png",
            # Damaged walls
            "game/assets-necessay/kenney/3D assets/Retro Urban Kit/Previews/wall-broken-type-a.png",
            "game/assets-necessay/kenney/3D assets/Retro Urban Kit/Previews/wall-broken-type-b.png",
            "game/assets-necessay/kenney/2D assets/Scribble Dungeons/PNG/Double (128px)/wall_damaged.png",
            "game/assets-necessay/kenney/2D assets/Scribble Dungeons/PNG/Double (128px)/wall_demolished.png",
            "game/assets-necessay/kenney/3D assets/Fantasy Town Kit/Previews/wallWoodBroken.png",
            "game/assets-necessay/kenney/3D assets/Fantasy Town Kit/Previews/wallBroken.png",
            # Audio
            "game/assets-necessay/kenney/Audio/Voiceover Pack Fighter/Audio/choose_your_character.ogg",
            # 更新音频文件后缀
            "game/assets-necessay/kenney/UI assets/UI Pack/Sounds/click-b.wav",
            "game/assets-necessay/kenney/Audio/Retro Sounds 2/Audio/explosion1.wav",
            "game/assets-necessay/kenney/Audio/Impact Sounds/Audio/footstep_wood_001.wav",
        ]
        self.load_assets()
    
    def load_assets(self):
        """Load all available assets"""
        # 尝试加载图片资源
        for path in self.asset_paths:
            if path.endswith(('.png', '.jpg', '.jpeg')):
                try:
                    if os.path.exists(path):
                        image = pygame.image.load(path)
                        filename = os.path.basename(path).lower()
                        
                        # 炸弹相关
                        if 'bomb' in filename and 'flash' not in filename:
                            if 'white' in filename:
                                self.images['bomb_white'] = pygame.transform.scale(image, (TILE_SIZE, TILE_SIZE))
                            else:
                                self.images['bomb'] = pygame.transform.scale(image, (TILE_SIZE, TILE_SIZE))
                        elif 'bombflash' in filename:
                            self.images['bomb_flash'] = pygame.transform.scale(image, (TILE_SIZE, TILE_SIZE))
                        
                        # 角色相关 - 使用统一的Abstract Platformer系列
                        elif 'playergreen_stand' in filename:
                            self.images['player_base'] = pygame.transform.scale(image, (TILE_SIZE-8, TILE_SIZE-8))
                        elif 'playerblue_stand' in filename:
                            self.images['player_blue_base'] = pygame.transform.scale(image, (TILE_SIZE-8, TILE_SIZE-8))
                        elif 'playergrey_stand' in filename:
                            self.images['player_grey_base'] = pygame.transform.scale(image, (TILE_SIZE-8, TILE_SIZE-8))
                        
                        # 备用角色图片
                        elif 'character' in filename or 'gamer' in filename or 'employee' in filename:
                            if 'gamer' in filename:
                                self.images['fallback_player1'] = pygame.transform.scale(image, (TILE_SIZE-8, TILE_SIZE-8))
                            elif 'employee' in filename:
                                self.images['fallback_player2'] = pygame.transform.scale(image, (TILE_SIZE-8, TILE_SIZE-8))
                        elif 'face_robot' in filename:
                            self.images['fallback_player3'] = pygame.transform.scale(image, (TILE_SIZE-8, TILE_SIZE-8))
                        
                        # 爆炸效果
                        elif 'explosion' in filename or 'fire' in filename:
                            if 'regularexplosion' in filename:
                                frame_num = filename.split('explosion')[-1].split('.')[0]
                                self.images[f'explosion_{frame_num}'] = pygame.transform.scale(image, (TILE_SIZE, TILE_SIZE))
                            elif filename == 'fire.png':
                                self.images['fire_icon'] = pygame.transform.scale(image, (TILE_SIZE//2, TILE_SIZE//2))
                            elif 'fire_01' in filename:
                                self.images['flame'] = pygame.transform.scale(image, (TILE_SIZE, TILE_SIZE))
                        
                        # 方块和墙体
                        elif 'block_' in filename:
                            block_num = filename.split('block_')[-1].split('.')[0]
                            self.images[f'block_{block_num}'] = pygame.transform.scale(image, (TILE_SIZE, TILE_SIZE))
                        elif 'wall' in filename:
                            if 'broken' in filename:
                                if 'type-a' in filename:
                                    self.images['wall_broken_a'] = pygame.transform.scale(image, (TILE_SIZE, TILE_SIZE))
                                elif 'type-b' in filename:
                                    self.images['wall_broken_b'] = pygame.transform.scale(image, (TILE_SIZE, TILE_SIZE))
                                elif 'wood' in filename:
                                    self.images['wall_wood_broken'] = pygame.transform.scale(image, (TILE_SIZE, TILE_SIZE))
                                else:
                                    self.images['wall_broken'] = pygame.transform.scale(image, (TILE_SIZE, TILE_SIZE))
                            elif 'damaged' in filename:
                                self.images['wall_damaged'] = pygame.transform.scale(image, (TILE_SIZE, TILE_SIZE))
                            elif 'demolished' in filename:
                                self.images['wall_demolished'] = pygame.transform.scale(image, (TILE_SIZE, TILE_SIZE))
                        
                        # 地板瓦片
                        elif 'tile_' in filename:
                            tile_num = filename.split('tile_')[-1].split('.')[0]
                            self.images[f'floor_{tile_num}'] = pygame.transform.scale(image, (TILE_SIZE, TILE_SIZE))
                        
                except Exception as e:
                    print(f"无法加载图片 {path}: {e}")
        
        # 尝试加载声音资源
        for path in self.asset_paths:
            if path.endswith(('.ogg', '.wav', '.mp3')):
                try:
                    if os.path.exists(path):
                        sound = pygame.mixer.Sound(path)
                        filename = os.path.basename(path).lower()
                        if 'choose' in filename:
                            self.sounds['game_start'] = sound
                        elif 'click-b' in filename:  # 炸弹放置音效
                            self.sounds['bomb_place'] = sound
                        elif 'explosion' in filename:  # 炸弹爆炸音效
                            self.sounds['bomb_explode'] = sound
                        elif 'footstep' in filename:  # 玩家行走音效
                            self.sounds['player_walk'] = sound
                except Exception as e:
                    print(f"无法加载声音 {path}: {e}")
        
        # Create player sprites from base image
        self.create_player_sprites()
        
        # Create default floor pattern
        self.create_floor_pattern()
    
    def create_player_sprites(self):
        """Create colored player sprites from base image"""
        # 获取基础玩家图片
        base_image = None
        
        # 优先使用Abstract Platformer系列的角色
        if 'player_base' in self.images:
            base_image = self.images['player_base']
        elif 'player_blue_base' in self.images:
            base_image = self.images['player_blue_base']
        elif 'player_grey_base' in self.images:
            base_image = self.images['player_grey_base']
        
        if base_image:
            # 创建四个不同风格的玩家精灵
            self.create_varied_players(base_image)
        else:
            # 如果没有找到基础图片，创建简单的几何图形玩家
            self.create_fallback_players()
    
    def create_varied_players(self, base_image):
        """Create four visually distinct player variants from base image"""
        player_configs = [
            {
                'color': (255, 80, 80),     # 红色 - Player 1
                'brightness': 1.2,
                'contrast': 1.1,
                'hue_shift': 0,
                'outline_color': (200, 0, 0),
                'effect': 'normal'
            },
            {
                'color': (80, 80, 255),     # 蓝色 - Player 2
                'brightness': 1.0,
                'contrast': 1.3,
                'hue_shift': 0,
                'outline_color': (0, 0, 200),
                'effect': 'metallic'
            },
            {
                'color': (80, 255, 80),     # 绿色 - Player 3
                'brightness': 1.1,
                'contrast': 1.0,
                'hue_shift': 0,
                'outline_color': (0, 200, 0),
                'effect': 'glow'
            },
            {
                'color': (255, 255, 80),    # 黄色 - Player 4
                'brightness': 1.3,
                'contrast': 0.9,
                'hue_shift': 0,
                'outline_color': (200, 200, 0),
                'effect': 'shadow'
            }
        ]
        
        for i, config in enumerate(player_configs):
            # 创建基础精灵副本
            sprite = base_image.copy()
            
            # 应用主要颜色调色
            color_overlay = pygame.Surface(sprite.get_size())
            color_overlay.fill(config['color'])
            sprite.blit(color_overlay, (0, 0), special_flags=pygame.BLEND_MULT)
            
            # 应用亮度调整
            if config['brightness'] != 1.0:
                brightness_overlay = pygame.Surface(sprite.get_size())
                brightness_value = int(255 * (config['brightness'] - 1.0))
                if brightness_value > 0:
                    brightness_overlay.fill((brightness_value, brightness_value, brightness_value))
                    sprite.blit(brightness_overlay, (0, 0), special_flags=pygame.BLEND_ADD)
                else:
                    brightness_overlay.fill((-brightness_value, -brightness_value, -brightness_value))
                    sprite.blit(brightness_overlay, (0, 0), special_flags=pygame.BLEND_SUB)
            
            # 应用特殊效果
            sprite = self.apply_player_effect(sprite, config['effect'], config['outline_color'])
            
            # 存储玩家精灵
            self.images[f'player{i+1}'] = sprite
    
    def apply_player_effect(self, sprite, effect, outline_color):
        """Apply special visual effects to player sprites"""
        if effect == 'metallic':
            # 金属效果 - 添加高光和反射
            enhanced_sprite = sprite.copy()
            highlight = pygame.Surface(sprite.get_size(), pygame.SRCALPHA)
            
            # 添加左上角高光
            w, h = sprite.get_size()
            for y in range(h//3):
                for x in range(w//3):
                    alpha = max(0, 100 - (x + y) * 2)
                    highlight.set_at((x, y), (255, 255, 255, alpha))
            
            enhanced_sprite.blit(highlight, (0, 0))
            return enhanced_sprite
            
        elif effect == 'glow':
            # 发光效果 - 添加外发光
            glowing_sprite = pygame.Surface((sprite.get_width() + 4, sprite.get_height() + 4), pygame.SRCALPHA)
            
            # 创建发光背景
            for offset in range(1, 3):
                glow_surface = pygame.Surface(sprite.get_size(), pygame.SRCALPHA)
                glow_surface.fill(outline_color + (80//offset,))
                for dx in range(-offset, offset+1):
                    for dy in range(-offset, offset+1):
                        if dx*dx + dy*dy <= offset*offset:
                            glowing_sprite.blit(glow_surface, (2+dx, 2+dy))
            
            # 叠加原始精灵
            glowing_sprite.blit(sprite, (2, 2))
            return glowing_sprite
            
        elif effect == 'shadow':
            # 阴影效果 - 添加深色轮廓
            shadow_sprite = pygame.Surface((sprite.get_width() + 2, sprite.get_height() + 2), pygame.SRCALPHA)
            
            # 创建阴影
            shadow_surface = pygame.Surface(sprite.get_size())
            shadow_surface.fill((30, 30, 30))
            shadow_surface.set_alpha(120)
            
            # 多方向阴影
            for dx in [-1, 0, 1]:
                for dy in [-1, 0, 1]:
                    if dx != 0 or dy != 0:
                        shadow_sprite.blit(shadow_surface, (1+dx, 1+dy))
            
            # 叠加原始精灵
            shadow_sprite.blit(sprite, (1, 1))
            return shadow_sprite
            
        else:  # normal
            # 普通效果 - 添加简单轮廓
            outlined_sprite = pygame.Surface((sprite.get_width() + 2, sprite.get_height() + 2), pygame.SRCALPHA)
            
            # 创建轮廓
            outline_surface = pygame.Surface(sprite.get_size())
            outline_surface.fill(outline_color)
            outline_surface.set_alpha(150)
            
            # 4方向轮廓
            for dx, dy in [(-1, 0), (1, 0), (0, -1), (0, 1)]:
                outlined_sprite.blit(outline_surface, (1+dx, 1+dy))
            
            # 叠加原始精灵
            outlined_sprite.blit(sprite, (1, 1))
            return outlined_sprite

    def create_fallback_players(self):
        """Create fallback geometric player sprites with distinct shapes"""
        player_configs = [
            {
                'color': RED,
                'shape': 'circle',
                'pattern': 'solid',
                'size_modifier': 1.0
            },
            {
                'color': BLUE,
                'shape': 'square',
                'pattern': 'striped',
                'size_modifier': 0.9
            },
            {
                'color': GREEN,
                'shape': 'triangle',
                'pattern': 'dotted',
                'size_modifier': 1.1
            },
            {
                'color': YELLOW,
                'shape': 'diamond',
                'pattern': 'checkered',
                'size_modifier': 1.0
            }
        ]
        
        for i, config in enumerate(player_configs):
            sprite_size = int((TILE_SIZE-8) * config['size_modifier'])
            player_surface = pygame.Surface((TILE_SIZE-8, TILE_SIZE-8), pygame.SRCALPHA)
            center = ((TILE_SIZE-8)//2, (TILE_SIZE-8)//2)
            
            # 绘制阴影
            shadow_offset = 2
            if config['shape'] == 'circle':
                pygame.draw.circle(player_surface, (50, 50, 50), 
                                 (center[0]+shadow_offset, center[1]+shadow_offset), sprite_size//3)
            elif config['shape'] == 'square':
                rect = pygame.Rect(center[0] - sprite_size//3 + shadow_offset, 
                                 center[1] - sprite_size//3 + shadow_offset, 
                                 sprite_size//3*2, sprite_size//3*2)
                pygame.draw.rect(player_surface, (50, 50, 50), rect)
            elif config['shape'] == 'triangle':
                points = [
                    (center[0] + shadow_offset, center[1] - sprite_size//3 + shadow_offset),
                    (center[0] - sprite_size//3 + shadow_offset, center[1] + sprite_size//3 + shadow_offset),
                    (center[0] + sprite_size//3 + shadow_offset, center[1] + sprite_size//3 + shadow_offset)
                ]
                pygame.draw.polygon(player_surface, (50, 50, 50), points)
            elif config['shape'] == 'diamond':
                points = [
                    (center[0] + shadow_offset, center[1] - sprite_size//3 + shadow_offset),
                    (center[0] + sprite_size//3 + shadow_offset, center[1] + shadow_offset),
                    (center[0] + shadow_offset, center[1] + sprite_size//3 + shadow_offset),
                    (center[0] - sprite_size//3 + shadow_offset, center[1] + shadow_offset)
                ]
                pygame.draw.polygon(player_surface, (50, 50, 50), points)
            
            # 绘制主体形状
            if config['shape'] == 'circle':
                pygame.draw.circle(player_surface, config['color'], center, sprite_size//3)
                if config['pattern'] == 'dotted':
                    for angle in range(0, 360, 45):
                        dot_x = center[0] + int((sprite_size//6) * math.cos(math.radians(angle)))
                        dot_y = center[1] + int((sprite_size//6) * math.sin(math.radians(angle)))
                        pygame.draw.circle(player_surface, WHITE, (dot_x, dot_y), 2)
                        
            elif config['shape'] == 'square':
                rect = pygame.Rect(center[0] - sprite_size//3, center[1] - sprite_size//3, 
                                 sprite_size//3*2, sprite_size//3*2)
                pygame.draw.rect(player_surface, config['color'], rect)
                if config['pattern'] == 'striped':
                    for i in range(0, sprite_size//3*2, 4):
                        line_rect = pygame.Rect(rect.x, rect.y + i, rect.width, 2)
                        pygame.draw.rect(player_surface, WHITE, line_rect)
                        
            elif config['shape'] == 'triangle':
                points = [
                    (center[0], center[1] - sprite_size//3),
                    (center[0] - sprite_size//3, center[1] + sprite_size//3),
                    (center[0] + sprite_size//3, center[1] + sprite_size//3)
                ]
                pygame.draw.polygon(player_surface, config['color'], points)
                
            elif config['shape'] == 'diamond':
                points = [
                    (center[0], center[1] - sprite_size//3),
                    (center[0] + sprite_size//3, center[1]),
                    (center[0], center[1] + sprite_size//3),
                    (center[0] - sprite_size//3, center[1])
                ]
                pygame.draw.polygon(player_surface, config['color'], points)
                if config['pattern'] == 'checkered':
                    # 添加棋盘格效果
                    for dx in range(-sprite_size//6, sprite_size//6, 4):
                        for dy in range(-sprite_size//6, sprite_size//6, 4):
                            if (dx//4 + dy//4) % 2:
                                check_rect = pygame.Rect(center[0] + dx, center[1] + dy, 3, 3)
                                pygame.draw.rect(player_surface, WHITE, check_rect)
            
            # 添加边框
            if config['shape'] == 'circle':
                pygame.draw.circle(player_surface, BLACK, center, sprite_size//3, 2)
            elif config['shape'] == 'square':
                rect = pygame.Rect(center[0] - sprite_size//3, center[1] - sprite_size//3, 
                                 sprite_size//3*2, sprite_size//3*2)
                pygame.draw.rect(player_surface, BLACK, rect, 2)
            elif config['shape'] == 'triangle':
                points = [
                    (center[0], center[1] - sprite_size//3),
                    (center[0] - sprite_size//3, center[1] + sprite_size//3),
                    (center[0] + sprite_size//3, center[1] + sprite_size//3)
                ]
                pygame.draw.polygon(player_surface, BLACK, points, 2)
            elif config['shape'] == 'diamond':
                points = [
                    (center[0], center[1] - sprite_size//3),
                    (center[0] + sprite_size//3, center[1]),
                    (center[0], center[1] + sprite_size//3),
                    (center[0] - sprite_size//3, center[1])
                ]
                pygame.draw.polygon(player_surface, BLACK, points, 2)
            
            # 添加玩家编号
            font = pygame.font.Font(None, 16)
            number_text = font.render(str(i+1), True, WHITE)
            number_rect = number_text.get_rect(center=center)
            player_surface.blit(number_text, number_rect)
            
            self.images[f'player{i+1}'] = player_surface
    
    def create_floor_pattern(self):
        """Create dark themed floor texture patterns"""
        # Create dark minimal floor - dark gray
        floor_surface = pygame.Surface((TILE_SIZE, TILE_SIZE))
        floor_surface.fill((45, 45, 45))  # Dark gray
        # Add subtle grid lines
        pygame.draw.line(floor_surface, (55, 55, 55), (TILE_SIZE-1, 0), (TILE_SIZE-1, TILE_SIZE-1), 1)
        pygame.draw.line(floor_surface, (55, 55, 55), (0, TILE_SIZE-1), (TILE_SIZE-1, TILE_SIZE-1), 1)
        self.images['clean_floor'] = floor_surface
        
        # Create alternative dark floor - slightly lighter
        floor_surface2 = pygame.Surface((TILE_SIZE, TILE_SIZE))
        floor_surface2.fill((50, 50, 50))
        pygame.draw.line(floor_surface2, (60, 60, 60), (TILE_SIZE-1, 0), (TILE_SIZE-1, TILE_SIZE-1), 1)
        pygame.draw.line(floor_surface2, (60, 60, 60), (0, TILE_SIZE-1), (TILE_SIZE-1, TILE_SIZE-1), 1)
        self.images['clean_floor_alt'] = floor_surface2
        
        # Create third dark variation
        floor_surface3 = pygame.Surface((TILE_SIZE, TILE_SIZE))
        floor_surface3.fill((40, 40, 40))
        pygame.draw.line(floor_surface3, (50, 50, 50), (TILE_SIZE-1, 0), (TILE_SIZE-1, TILE_SIZE-1), 1)
        pygame.draw.line(floor_surface3, (50, 50, 50), (0, TILE_SIZE-1), (TILE_SIZE-1, TILE_SIZE-1), 1)
        self.images['clean_floor_bright'] = floor_surface3
        
        # Create checkered dark floor pattern
        floor_surface4 = pygame.Surface((TILE_SIZE, TILE_SIZE))
        floor_surface4.fill((42, 42, 42))
        # Add subtle checkered pattern
        for i in range(0, TILE_SIZE, 16):
            for j in range(0, TILE_SIZE, 16):
                if (i//16 + j//16) % 2:
                    check_rect = pygame.Rect(i, j, 16, 16)
                    pygame.draw.rect(floor_surface4, (48, 48, 48), check_rect)
        self.images['clean_floor_checkered'] = floor_surface4
        
        # Create improved hard wall texture
        hard_wall_surface = pygame.Surface((TILE_SIZE, TILE_SIZE))
        hard_wall_surface.fill((90, 90, 90))
        # Add simple brick pattern
        for i in range(0, TILE_SIZE, 16):
            for j in range(0, TILE_SIZE, 8):
                offset = 8 if (j // 8) % 2 else 0
                brick_rect = pygame.Rect(i + offset, j, 14, 6)
                pygame.draw.rect(hard_wall_surface, (110, 110, 110), brick_rect)
                pygame.draw.rect(hard_wall_surface, (70, 70, 70), brick_rect, 1)
        self.images['default_hard_wall'] = hard_wall_surface
        
        # Create soft wall texture
        soft_wall_surface = pygame.Surface((TILE_SIZE, TILE_SIZE))
        soft_wall_surface.fill((139, 90, 43))  # Brown
        # Add wood grain effect
        for i in range(0, TILE_SIZE, 8):
            for j in range(0, TILE_SIZE, 8):
                wood_rect = pygame.Rect(i, j, 6, 6)
                pygame.draw.rect(soft_wall_surface, (160, 110, 60), wood_rect)
                pygame.draw.rect(soft_wall_surface, (120, 70, 30), wood_rect, 1)
        self.images['default_soft_wall'] = soft_wall_surface
    
    def get_image(self, name: str) -> Optional[pygame.Surface]:
        """获取图片资源"""
        return self.images.get(name)
    
    def get_sound(self, name: str) -> Optional[pygame.mixer.Sound]:
        """获取声音资源"""
        return self.sounds.get(name)
    
    def get_random_floor_tile(self, x: int, y: int) -> pygame.Surface:
        """Get random dark floor tile based on position"""
        # Use position as seed to ensure same tile at same position
        random.seed(x * 1000 + y)
        
        # Use only dark floor variants
        dark_floors = ['clean_floor', 'clean_floor_alt', 'clean_floor_bright', 'clean_floor_checkered']
        tile_name = random.choice(dark_floors)
        return self.images[tile_name]
    
    def get_wall_texture(self, wall_type: str, x: int, y: int) -> pygame.Surface:
        """Get wall texture based on type and position"""
        random.seed(x * 1000 + y)
        
        if wall_type == 'soft':
            # Soft walls use wood or block textures
            soft_wall_options = ['wall_wood_broken', 'block_01', 'block_02', 'block_03']
            for option in soft_wall_options:
                if option in self.images:
                    return self.images[option]
            return self.images['default_soft_wall']
        elif wall_type == 'hard':
            # Hard walls use stone textures
            hard_wall_options = ['block_04', 'block_05', 'block_06', 'block_07', 'block_08']
            for option in hard_wall_options:
                if option in self.images:
                    return self.images[option]
            return self.images['default_hard_wall']
        
        # Default texture
        return self.images.get('default_hard_wall')

class GameState:
    """游戏状态管理器"""
    
    def __init__(self, difficulty=DifficultyLevel.NORMAL):
        # 保存难度设置
        self.difficulty = difficulty
        self.difficulty_config = get_difficulty_config(difficulty)
        
        # 基于难度配置设置网格大小
        global GRID_WIDTH, GRID_HEIGHT
        GRID_WIDTH = self.difficulty_config.grid_width
        GRID_HEIGHT = self.difficulty_config.grid_height
        
        self.grid = [[Tile.FLOOR for _ in range(GRID_WIDTH)] for _ in range(GRID_HEIGHT)]
        self.players: List[Player] = []
        self.bombs: List[Bomb] = []
        self.items: List[ItemDrop] = []
        self.flames = {}  # 修改为字典: (x, y) -> 剩余时间
        self.flame_duration = 60  # 火焰持续帧数
        self.game_over = False
        self.winner = None
        self.asset_manager = AssetManager()
        
        # 爆炸回调函数
        self.register_explosion_callback = None
        self.register_kill_callback = None
        self.register_item_pickup_callback = None
    
        # 初始化地图
        self.init_arena()
        
        # 初始化玩家
        self.init_players()
    
    def init_arena(self):
        """初始化竞技场地图"""
        # 放置硬墙在奇数行奇数列交叉点
        for y in range(1, GRID_HEIGHT, 2):
            for x in range(1, GRID_WIDTH, 2):
                self.grid[y][x] = Tile.HARD
        
        # 根据难度配置随机放置软墙
        soft_wall_chance = self.difficulty_config.soft_wall_chance
        clear_radius = self.difficulty_config.clear_radius
        
        for y in range(GRID_HEIGHT):
            for x in range(GRID_WIDTH):
                if self.grid[y][x] == Tile.FLOOR:
                    # 跳过四个角落的clear_radius×clear_radius区域(玩家出生点)
                    if ((x < clear_radius and y < clear_radius) or 
                        (x >= GRID_WIDTH-clear_radius and y < clear_radius) or
                        (x < clear_radius and y >= GRID_HEIGHT-clear_radius) or
                        (x >= GRID_WIDTH-clear_radius and y >= GRID_HEIGHT-clear_radius)):
                        continue
                    
                    if random.random() < soft_wall_chance:
                        self.grid[y][x] = Tile.SOFT
    
    def init_players(self):
        """初始化玩家"""
        # 玩家出生位置(四个角落)
        spawn_positions = [
            Vector2(0, 0),                              # 左上
            Vector2(GRID_WIDTH-1, 0),                   # 右上
            Vector2(0, GRID_HEIGHT-1),                  # 左下
            Vector2(GRID_WIDTH-1, GRID_HEIGHT-1)        # 右下
        ]
        
        # 输入方案
        input_schemes = [
            {'up': pygame.K_w, 'down': pygame.K_s, 'left': pygame.K_a, 'right': pygame.K_d, 'bomb': pygame.K_SPACE},
            {'up': pygame.K_UP, 'down': pygame.K_DOWN, 'left': pygame.K_LEFT, 'right': pygame.K_RIGHT, 'bomb': pygame.K_RETURN},
            {'up': pygame.K_t, 'down': pygame.K_g, 'left': pygame.K_f, 'right': pygame.K_h, 'bomb': pygame.K_r},
            {'up': pygame.K_i, 'down': pygame.K_k, 'left': pygame.K_j, 'right': pygame.K_l, 'bomb': pygame.K_u}
        ]
        
        # 玩家颜色
        colors = [RED, BLUE, GREEN, YELLOW]
        
        for i in range(4):
            player = Player(
                pos=spawn_positions[i],
                input_scheme=input_schemes[i],
                color=colors[i],
                player_id=i,
                speed=self.difficulty_config.player_speed  # 使用难度配置的速度
            )
            self.players.append(player)

    def handle_input(self, keys):
        """处理玩家输入"""
        for player in self.players:
            if not player.alive:
                continue
            
            # 更新移动冷却时间
            if player.move_cooldown > 0:
                player.move_cooldown -= 1
                continue
            
            # 移动输入
            new_pos = Vector2(player.pos.x, player.pos.y)
            moved = False
            
            if keys[player.input_scheme['up']]:
                new_pos.y -= 1
                moved = True
                print("PLAYER_MOVED")
            elif keys[player.input_scheme['down']]:
                new_pos.y += 1
                moved = True
                print("PLAYER_MOVED")
            elif keys[player.input_scheme['left']]:
                new_pos.x -= 1
                moved = True
                print("PLAYER_MOVED")
            elif keys[player.input_scheme['right']]:
                new_pos.x += 1
                moved = True
                print("PLAYER_MOVED")
            
            # 检查移动是否有效
            if moved and self.is_valid_move(new_pos):
                player.pos = new_pos
                # 设置移动冷却时间，基于玩家速度调整
                base_cooldown = MOVE_DELAY
                speed_modifier = max(1, player.speed)
                player.move_cooldown = max(1, base_cooldown - (speed_modifier - 1) * 2)
                
                # 播放行走音效
                player_walk_sound = self.asset_manager.get_sound('player_walk')
                if player_walk_sound:
                    player_walk_sound.play()
                
                # 检查道具拾取
                self.check_item_pickup(player)
            
            # 炸弹放置输入
            if keys[player.input_scheme['bomb']]:
                self.place_bomb(player)
    
    def is_valid_move(self, pos: Vector2) -> bool:
        """检查移动是否有效"""
        # 边界检查
        if pos.x < 0 or pos.x >= GRID_WIDTH or pos.y < 0 or pos.y >= GRID_HEIGHT:
            return False
        
        # 障碍物检查
        tile = self.grid[pos.y][pos.x]
        if tile in [Tile.HARD, Tile.SOFT, Tile.BOMB]:
            return False
        
        return True
    
    def place_bomb(self, player: Player):
        """放置炸弹"""
        # 检查炸弹数量限制
        if player.active_bombs >= player.bombs:
            return
        
        # 检查当前位置是否已有炸弹
        bomb_pos = player.pos
        for bomb in self.bombs:
            if bomb.grid_pos.x == bomb_pos.x and bomb.grid_pos.y == bomb_pos.y:
                return
        
        # 放置炸弹
        bomb = Bomb(
            owner_id=player.player_id,
            grid_pos=Vector2(bomb_pos.x, bomb_pos.y),
            fire=player.fire
        )
        self.bombs.append(bomb)
        self.grid[bomb_pos.y][bomb_pos.x] = Tile.BOMB
        player.active_bombs += 1
        print("BOMB_PLACED")
        
        # 播放炸弹放置音效
        bomb_place_sound = self.asset_manager.get_sound('bomb_place')
        if bomb_place_sound:
            bomb_place_sound.play()
    
    def update_bombs(self):
        """更新炸弹状态"""
        bombs_to_remove = []
        
        for bomb in self.bombs:
            bomb.timer -= 1
            
            if bomb.timer <= 0:
                self.explode_bomb(bomb)
                bombs_to_remove.append(bomb)
        
        # 移除爆炸的炸弹
        for bomb in bombs_to_remove:
            self.bombs.remove(bomb)
            self.grid[bomb.grid_pos.y][bomb.grid_pos.x] = Tile.FLOOR
            # 减少玩家活跃炸弹数量
            if bomb.owner_id < len(self.players):
                self.players[bomb.owner_id].active_bombs -= 1
    
    def explode_bomb(self, bomb: Bomb):
        """炸弹爆炸处理"""
        print("BOMB_EXPLODED")
        
        # 播放爆炸音效
        bomb_explode_sound = self.asset_manager.get_sound('bomb_explode')
        if bomb_explode_sound:
            bomb_explode_sound.play()
        
        # 爆炸中心
        center_x, center_y = bomb.grid_pos.x, bomb.grid_pos.y
        self.flames[(center_x, center_y)] = self.flame_duration
        
        # 记录所有受影响的位置
        affected_positions = {(center_x, center_y)}
        
        # 四个方向的火焰传播
        directions = [(0, 1), (0, -1), (1, 0), (-1, 0)]  # 下、上、右、左
        
        for dx, dy in directions:
            for i in range(1, bomb.fire + 1):
                flame_x = center_x + dx * i
                flame_y = center_y + dy * i
                
                # 边界检查
                if flame_x < 0 or flame_x >= GRID_WIDTH or flame_y < 0 or flame_y >= GRID_HEIGHT:
                    break
                
                tile = self.grid[flame_y][flame_x]
                
                # 硬墙阻挡火焰
                if tile == Tile.HARD:
                    break
                
                # 添加火焰
                self.flames[(flame_x, flame_y)] = self.flame_duration
                affected_positions.add((flame_x, flame_y))
                
                # 软墙被摧毁并阻挡火焰
                if tile == Tile.SOFT:
                    self.grid[flame_y][flame_x] = Tile.FLOOR
                    # 使用难度配置的道具掉落概率
                    item_drop_chance = self.difficulty_config.item_drop_chance
                    if random.random() < item_drop_chance:
                        item_type = random.choice(list(Item))
                        self.items.append(ItemDrop(item_type, Vector2(flame_x, flame_y)))
                    break
            
                # 炸弹连锁反应
                if tile == Tile.BOMB:
                    for other_bomb in self.bombs:
                        if (other_bomb.grid_pos.x == flame_x and 
                            other_bomb.grid_pos.y == flame_y):
                            other_bomb.timer = 0  # 立即爆炸
                            break
    
        # 如果存在爆炸回调，调用它
        if self.register_explosion_callback:
            self.register_explosion_callback(bomb, affected_positions)
        
        # 检查玩家是否被火焰击中
        self.check_flame_damage()
    
    def update_flames(self):
        """更新火焰状态"""
        expired_flames = []
        
        # 更新每个火焰位置的计时器
        for pos, timer in self.flames.items():
            # 减少计时器
            self.flames[pos] = timer - 1
            
            # 如果计时器归零，标记为过期
            if self.flames[pos] <= 0:
                expired_flames.append(pos)
        
        # 移除过期火焰
        for pos in expired_flames:
            self.flames.pop(pos)
    
    def check_flame_damage(self):
        """检查火焰伤害"""
        for player in self.players:
            if not player.alive:
                continue
            
            player_pos = (player.pos.x, player.pos.y)
            if player_pos in self.flames:
                if player.trapped_ticks > 0:
                    # 已经被困住，直接死亡
                    player.alive = False
                    
                    # 寻找最近的导致死亡的炸弹
                    killer_id = None
                    min_distance = float('inf')
                    
                    # 遍历所有火焰位置，寻找导致死亡的炸弹
                    for bomb in self.bombs:
                        # 计算曼哈顿距离
                        distance = abs(bomb.grid_pos.x - player.pos.x) + abs(bomb.grid_pos.y - player.pos.y)
                        if distance <= bomb.fire and distance < min_distance:
                            min_distance = distance
                            killer_id = bomb.owner_id
                    
                    # 如果找到了炸弹，触发击杀回调
                    if killer_id is not None and self.register_kill_callback:
                        self.register_kill_callback(killer_id, player.player_id)
                else:
                    # 第一次被困住
                    player.trapped_ticks = 150  # 2.5秒
    
    def update_players(self):
        """更新玩家状态"""
        for player in self.players:
            if not player.alive:
                continue
            
            # 更新被困状态
            if player.trapped_ticks > 0:
                player.trapped_ticks -= 1
                if player.trapped_ticks <= 0:
                    # 检查是否有队友救援
                    rescued = False
                    for other_player in self.players:
                        if (other_player != player and other_player.alive and
                            other_player.pos.x == player.pos.x and
                            other_player.pos.y == player.pos.y):
                            rescued = True
                            break
                    
                    if not rescued:
                        player.alive = False
    
    def check_item_pickup(self, player: Player):
        """检查道具拾取"""
        items_to_remove = []
        
        for item in self.items:
            if (item.grid_pos.x == player.pos.x and 
                item.grid_pos.y == player.pos.y):
                
                # 应用道具效果
                if item.item_type == Item.FIRE_UP:
                    player.fire = min(player.fire + 1, 8)
                elif item.item_type == Item.BOMB_UP:
                    player.bombs = min(player.bombs + 1, 8)
                elif item.item_type == Item.SPEED_UP:
                    player.speed = min(player.speed + 1, 8)  # 速度提升后会增加最大移动距离
            
                items_to_remove.append(item)
                print("ITEM_COLLECTED")
                
                # 触发道具拾取回调
                if self.register_item_pickup_callback:
                    self.register_item_pickup_callback(player.player_id, item.item_type)
                
                # 播放拾取音效
                pickup_sound = self.asset_manager.get_sound('pickup')
                if pickup_sound:
                    pickup_sound.play()
        
        # 移除已拾取的道具
        for item in items_to_remove:
            self.items.remove(item)
    
    def check_win_condition(self):
        """检查胜利条件"""
        alive_players = [p for p in self.players if p.alive]
        
        if len(alive_players) <= 1:
            self.game_over = True
            self.winner = alive_players[0] if alive_players else None
            print("GAME_OVER")
    
    def update(self):
        """更新游戏状态"""
        if self.game_over:
            return
        
        self.update_bombs()
        self.update_flames()
        self.update_players()
        self.check_win_condition()
    
    def reset(self, difficulty=None):
        """重置游戏"""
        if difficulty is None:
            difficulty = self.difficulty
        self.__init__(difficulty=difficulty)

    def to_dict(self):
        """将游戏状态转换为字典，用于序列化"""
        state = {
            "grid": [[tile.value for tile in row] for row in self.grid],
            "players": [],
            "bombs": [],
            "items": [],
            "flames": list(self.flames),
            "flame_timer": self.flame_timer,
            "game_over": self.game_over,
            "winner": self.winner.player_id if self.winner else None
        }
        
        # 序列化玩家
        for player in self.players:
            player_data = {
                "pos_x": player.pos.x,
                "pos_y": player.pos.y,
                "fire": player.fire,
                "bombs": player.bombs,
                "speed": player.speed,
                "alive": player.alive,
                "trapped_ticks": player.trapped_ticks,
                "input_scheme": player.input_scheme,
                "color": player.color,
                "active_bombs": player.active_bombs,
                "player_id": player.player_id,
                "move_cooldown": player.move_cooldown
            }
            state["players"].append(player_data)
        
        # 序列化炸弹
        for bomb in self.bombs:
            bomb_data = {
                "owner_id": bomb.owner_id,
                "grid_pos_x": bomb.grid_pos.x,
                "grid_pos_y": bomb.grid_pos.y,
                "timer": bomb.timer,
                "fire": bomb.fire
            }
            state["bombs"].append(bomb_data)
        
        # 序列化道具
        for item in self.items:
            item_data = {
                "item_type": item.item_type.value,
                "grid_pos_x": item.grid_pos.x,
                "grid_pos_y": item.grid_pos.y
            }
            state["items"].append(item_data)
            
        return state
    
    @classmethod
    def from_dict(cls, state_dict, asset_manager=None):
        """从字典创建游戏状态实例"""
        state = cls()
        
        # 如果提供了asset_manager，使用它
        if asset_manager:
            state.asset_manager = asset_manager
            
        # 恢复grid
        for y, row in enumerate(state_dict["grid"]):
            for x, cell in enumerate(row):
                state.grid[y][x] = Tile(cell)
        
        # 清空当前的玩家、炸弹和道具列表
        state.players = []
        state.bombs = []
        state.items = []
        
        # 恢复玩家
        for player_data in state_dict["players"]:
            player = Player(
                pos=Vector2(player_data["pos_x"], player_data["pos_y"]),
                fire=player_data["fire"],
                bombs=player_data["bombs"],
                speed=player_data["speed"],
                alive=player_data["alive"],
                trapped_ticks=player_data["trapped_ticks"],
                input_scheme=player_data["input_scheme"],
                color=player_data["color"],
                active_bombs=player_data["active_bombs"],
                player_id=player_data["player_id"],
                move_cooldown=player_data["move_cooldown"]
            )
            state.players.append(player)
        
        # 恢复炸弹
        for bomb_data in state_dict["bombs"]:
            bomb = Bomb(
                owner_id=bomb_data["owner_id"],
                grid_pos=Vector2(bomb_data["grid_pos_x"], bomb_data["grid_pos_y"]),
                timer=bomb_data["timer"],
                fire=bomb_data["fire"]
            )
            state.bombs.append(bomb)
        
        # 恢复道具
        for item_data in state_dict["items"]:
            item = ItemDrop(
                item_type=Item(item_data["item_type"]),
                grid_pos=Vector2(item_data["grid_pos_x"], item_data["grid_pos_y"])
            )
            state.items.append(item)
        
        # 恢复其他状态
        state.flames = set(tuple(pos) for pos in state_dict["flames"])
        state.flame_timer = state_dict["flame_timer"]
        state.game_over = state_dict["game_over"]
        
        # 恢复winner
        if state_dict["winner"] is not None:
            for player in state.players:
                if player.player_id == state_dict["winner"]:
                    state.winner = player
                    break
        else:
            state.winner = None
            
        return state

    # 为了支持步数倒计时而不是帧数倒计时，增加一个更新炸弹的方法变体
    def update_bombs_steps(self):
        """更新炸弹状态（使用步数计时）"""
        bombs_to_remove = []
        
        for bomb in self.bombs:
            bomb.timer -= 1
            
            if bomb.timer <= 0:
                self.explode_bomb(bomb)
                bombs_to_remove.append(bomb)
        
        # 移除爆炸的炸弹
        for bomb in bombs_to_remove:
            self.bombs.remove(bomb)
            self.grid[bomb.grid_pos.y][bomb.grid_pos.x] = Tile.FLOOR
            # 减少玩家活跃炸弹数量
            if bomb.owner_id < len(self.players):
                self.players[bomb.owner_id].active_bombs -= 1

class Renderer:
    """Renderer 类 - 负责游戏渲染"""
    
    def __init__(self, screen, game_state: GameState):
        self.screen = screen
        self.game_state = game_state
        self.font = pygame.font.Font(None, 36)
        self.small_font = pygame.font.Font(None, 24)
        self.explosion_frame = 0  # 爆炸动画帧
    
    def render(self):
        """渲染游戏画面"""
        # Dark background to match dark floor theme
        self.screen.fill((25, 25, 25))  # Very dark gray background
        
        # 渲染地图
        self.render_grid()
        
        # 渲染道具
        self.render_items()
        
        # 渲染炸弹
        self.render_bombs()
        
        # 渲染火焰
        self.render_flames()
        
        # 渲染玩家
        self.render_players()
        
        # 渲染UI
        self.render_ui()
        
        # 渲染游戏结束画面
        if self.game_state.game_over:
            self.render_game_over()
        
        # 更新动画帧
        self.explosion_frame = (self.explosion_frame + 1) % 240  # 4秒循环
        
        pygame.display.flip()
    
    def render_grid(self):
        """渲染地图网格"""
        for y in range(GRID_HEIGHT):
            for x in range(GRID_WIDTH):
                tile = self.game_state.grid[y][x]
                
                if tile == Tile.FLOOR:
                    # 使用随机的地板瓦片
                    floor_tile = self.game_state.asset_manager.get_random_floor_tile(x, y)
                    self.screen.blit(floor_tile, (x * TILE_SIZE, y * TILE_SIZE))
                elif tile == Tile.SOFT:
                    # 软墙使用木材纹理
                    soft_wall_texture = self.game_state.asset_manager.get_wall_texture('soft', x, y)
                    self.screen.blit(soft_wall_texture, (x * TILE_SIZE, y * TILE_SIZE))
                elif tile == Tile.HARD:
                    # 硬墙使用石材纹理
                    hard_wall_texture = self.game_state.asset_manager.get_wall_texture('hard', x, y)
                    self.screen.blit(hard_wall_texture, (x * TILE_SIZE, y * TILE_SIZE))
    
    def render_items(self):
        """渲染道具"""
        for item in self.game_state.items:
            item_image = self.game_state.asset_manager.get_image(item.item_type.name.lower())
            if item_image:
                self.screen.blit(item_image, (item.grid_pos.x * TILE_SIZE, item.grid_pos.y * TILE_SIZE))
    
    def render_bombs(self):
        """渲染炸弹"""
        for bomb in self.game_state.bombs:
            bomb_image = self.game_state.asset_manager.get_image('bomb')
            if bomb_image:
                self.screen.blit(bomb_image, (bomb.grid_pos.x * TILE_SIZE, bomb.grid_pos.y * TILE_SIZE))
    
    def render_flames(self):
        """渲染火焰"""
        flame_images = [self.game_state.asset_manager.get_image(f'explosion_{i:02}') for i in range(4)]
        
        for (flame_x, flame_y) in self.game_state.flames:
            for image in flame_images:
                if image:
                    self.screen.blit(image, (flame_x * TILE_SIZE, flame_y * TILE_SIZE))
    
    def render_players(self):
        """渲染玩家"""
        for player in self.game_state.players:
            if not player.alive:
                continue
            
            player_image = self.game_state.asset_manager.get_image(f'player{player.player_id+1}')
            if player_image:
                self.screen.blit(player_image, (player.pos.x * TILE_SIZE, player.pos.y * TILE_SIZE))
    
    def render_ui(self):
        """渲染用户界面"""
        for i, player in enumerate(self.game_state.players):
            if not player.alive:
                continue
            
            # 显示玩家信息
            info_text = f"Player {i+1}: Bombs={player.bombs} Fire={player.fire} Speed={player.speed}"
            text_surface = self.font.render(info_text, True, WHITE)
            self.screen.blit(text_surface, (10, 10 + i * 30))
        
        # 显示游戏结束信息
        if self.game_state.game_over:
            if self.game_state.winner:
                winner_text = f"Player {self.game_state.winner.player_id+1} Wins!"
                text_surface = self.font.render(winner_text, True, WHITE)
                self.screen.blit(text_surface, (WINDOW_WIDTH//2 - text_surface.get_width()//2, WINDOW_HEIGHT//2 - 20))
            
            restart_text = "Press R to Restart"
            text_surface = self.font.render(restart_text, True, WHITE)
            self.screen.blit(text_surface, (WINDOW_WIDTH//2 - text_surface.get_width()//2, WINDOW_HEIGHT//2 + 20))
    
    def render_game_over(self):
        """渲染游戏结束画面"""
        game_over_surface = pygame.Surface((WINDOW_WIDTH, WINDOW_HEIGHT))
        game_over_surface.set_alpha(200)  # 半透明黑色背景
        
        # 绘制游戏结束背景
        pygame.draw.rect(game_over_surface, (0, 0, 0), (0, 0, WINDOW_WIDTH, WINDOW_HEIGHT))
        
        # 绘制文本
        font = pygame.font.Font(None, 74)
        text = "Game Over"
        text_surface = font.render(text, True, WHITE)
        text_rect = text_surface.get_rect(center=(WINDOW_WIDTH//2, WINDOW_HEIGHT//2 - 50))
        game_over_surface.blit(text_surface, text_rect)
        
        if self.game_state.winner:
            winner_text = f"Player {self.game_state.winner.player_id+1} Wins!"
            winner_surface = font.render(winner_text, True, WHITE)
            winner_rect = winner_surface.get_rect(center=(WINDOW_WIDTH//2, WINDOW_HEIGHT//2 + 10))
            game_over_surface.blit(winner_surface, winner_rect)
        
        # 绘制重新开始提示
        font_small = pygame.font.Font(None, 36)
        restart_text = "Press R to Restart"
        restart_surface = font_small.render(restart_text, True, WHITE)
        restart_rect = restart_surface.get_rect(center=(WINDOW_WIDTH//2, WINDOW_HEIGHT//2 + 70))
        game_over_surface.blit(restart_surface, restart_rect)
        
        # 渲染到屏幕
        self.screen.blit(game_over_surface, (0, 0))

def main():
    """主函数"""
    # 允许选择难度
    difficulty = DifficultyLevel.NORMAL  # 默认难度
    
    # 简单的控制台选择界面
    print("请选择游戏难度:")
    print("1. 简单 (小地图，少障碍物)")
    print("2. 普通 (标准地图)")
    print("3. 困难 (大地图，更多障碍物)")
    choice = input("请输入选择 (1/2/3): ")
    
    if choice == '1':
        difficulty = DifficultyLevel.EASY
    elif choice == '3':
        difficulty = DifficultyLevel.HARD
    
    # 初始化Pygame
    pygame.init()
    
    # 初始化游戏状态和渲染器（使用选择的难度）
    game_state = GameState(difficulty=difficulty)
    
    # 根据选择的难度调整窗口大小
    screen = pygame.display.set_mode((WINDOW_WIDTH, WINDOW_HEIGHT))
    pygame.display.set_caption(f"Classic Bomberman - {game_state.difficulty_config.name}模式")
    
    clock = pygame.time.Clock()
    renderer = Renderer(screen, game_state)
    
    # 游戏主循环
    running = True
    while running:
        # 事件处理
        for event in pygame.event.get():
            if event.type == pygame.QUIT:
                running = False
            elif event.type == pygame.KEYDOWN:
                if event.key == pygame.K_r and game_state.game_over:
                    # 重新开始游戏
                    game_state.reset()
                    renderer.game_state = game_state
        
        # 输入处理
        keys = pygame.key.get_pressed()
        game_state.handle_input(keys)
        
        # 更新游戏状态
        game_state.update()
        
        # 渲染
        renderer.render()
        
        # 控制帧率
        clock.tick(FPS)
    
    pygame.quit()

if __name__ == '__main__':
    main()